//------------------------------------------------------------------------------
// File: cmn_badges.cs
// Since badges are currently global (not client or server based), the 
// functionality for them have been placed into this file, which includes badge
// creation and badge rendering
// Author: Matthew Rudge
//------------------------------------------------------------------------------

//-Globals----------------------------------------------------------------------
// Group ids determine which image to display when the badge has been awarded.
// You can name these however you want and create as many as you want.
$TSB_GROUPWHEEL = 0;
$TSB_GROUPWOOD = 1;
$TSB_GROUPPICK = 2;
$TSB_GROUPGUN = 3;
$TSB_GROUPHAT = 4;
$TSB_GROUPGOLD = 5;
$TSB_GROUPFOOD = 6;
$TSB_GROUPSKULL = 7;
$TSB_GROUPSHOE = 8;
$TSB_GROUPBOOT = 9;

// Render ids determine the order of the badges to be rendered on the badge page
$TSB_RENDERID = 0;

// Number of badges per page
$TSB_BADGEPERPAGE = 8;

// Number of pages and current page
$TSB_MAXPAGES = 3;
$TSB_CURRPAGE = 0;

// Used for forming badge strings to manipulate gui elements
$TSB_ICONGUI = "BadgeIcon";
$TSB_NAMEGUI = "BadgeName";
$TSB_DESCGUI = "BadgeDescription";
$TSB_CMPLGUI = "BadgeComplete";

//------------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Creates all the badges. This should only be done once per game session. The
//! first parameter is the string id for a badge, which is used to obtain
//! the name of the badge and the description for the badge from the task string
//! table (for instance, if the badge string id is foo, then the badge name 
//! string id is foo_name and the badge description string id is foo_desc).
////////////////////////////////////////////////////////////////////////////////
function tsCreateBadges()
{
   
   tsCreateBadge("badge_settler", $TSB_GROUPHAT, 0, false);          // done
   tsCreateBadge("badge_sheriff", $TSB_GROUPGUN, 1, false);          // done   
   tsCreateBadge("badge_farmer", $TSB_GROUPFOOD, 2, false);          // done
   tsCreateBadge("badge_rancher", $TSB_GROUPFOOD, 3, false);         // done
   tsCreateBadge("badge_foodking", $TSB_GROUPFOOD, 4, false);        // done
   tsCreateBadge("badge_carpenter", $TSB_GROUPWOOD, 5, false);       // done
   tsCreateBadge("badge_population", $TSB_GROUPBOOT, 6, false);      // done
   tsCreateBadge("badge_paradise", $TSB_GROUPWHEEL, 7, false);       // done
   tsCreateBadge("badge_gold", $TSB_GROUPGOLD, 8, false);            // done
   tsCreateBadge("badge_hub", $TSB_GROUPWHEEL, 9, false);            // done
   tsCreateBadge("badge_treasure", $TSB_GROUPGOLD, 10, true);        // done
   tsCreateBadge("badge_pioneer3", $TSB_GROUPBOOT, 11, false);       // done
   tsCreateBadge("badge_pioneer1", $TSB_GROUPHAT, 12, false);        // done
   tsCreateBadge("badge_mayor", $TSB_GROUPHAT, 13, false);           // done
   tsCreateBadge("badge_railroad", $TSB_GROUPWHEEL, 14, false);      // done
   tsCreateBadge("badge_bandit", $TSB_GROUPGUN, 15, false);          // done
   tsCreateBadge("badge_happy", $TSB_GROUPSHOE, 16, false);          // done
   tsCreateBadge("badge_city", $TSB_GROUPBOOT, 17, false);           // done
   tsCreateBadge("badge_food", $TSB_GROUPFOOD, 18, false);           // done
   tsCreateBadge("badge_experience", $TSB_GROUPPICK, 19, false);     // done
   tsCreateBadge("badge_sandboxchallenge", $TSB_GROUPSKULL, 20, false); // done
   tsCreateBadge("badge_honeypot", $TSB_GROUPSHOE, 21, true);        // done
   tsCreateBadge("badge_3amigos", $TSB_GROUPGUN, 22, true);          // done
   tsCreateBadge("badge_sheep", $TSB_GROUPSHOE, 23, true);           // done
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the normal badge image when the badge has not been awarded
//! \param %groupId Group id
//! \retval string Image path
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeNormalImage(%groupId)
{
   // Return normal image
   switch(%groupId) {
      case $TSB_GROUPWHEEL:
         return "assets/client/ui/hud/trophy_wheel.jpg";
      case $TSB_GROUPWOOD:
         return "assets/client/ui/hud/trophy_axe.jpg";
      case $TSB_GROUPPICK:
         return "assets/client/ui/hud/trophy_pick.jpg";
      case $TSB_GROUPGUN:
         return "assets/client/ui/hud/trophy_gun.jpg";
      case $TSB_GROUPHAT:
         return "assets/client/ui/hud/trophy_hat.jpg";
      case $TSB_GROUPGOLD:
         return "assets/client/ui/hud/trophy_gold.jpg";
      case $TSB_GROUPFOOD:
         return "assets/client/ui/hud/trophy_food.jpg";
      case $TSB_GROUPSKULL:
         return "assets/client/ui/hud/trophy_skull.jpg";
      case $TSB_GROUPSHOE:
         return "assets/client/ui/hud/trophy_shoe.jpg";
      case $TSB_GROUPBOOT:
         return "assets/client/ui/hud/trophy_boot.jpg";
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the locked badge image
//! \retval string Image path
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeLockedImage()
{
   return "assets/client/ui/hud/trophy_questionmark.jpg";
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the badge icon image to use based on the group id of the badge
//! \param %groupId Group id
//! \retval string Image path
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeAwardImage(%groupId)
{
   // Return awarded image
   switch(%groupId) {
      case $TSB_GROUPWHEEL:
         return "assets/client/ui/hud/trophy_wheelwon.jpg";
      case $TSB_GROUPWOOD:
         return "assets/client/ui/hud/trophy_axewon.jpg";
      case $TSB_GROUPPICK:
         return "assets/client/ui/hud/trophy_pickwon.jpg";
      case $TSB_GROUPGUN:
         return "assets/client/ui/hud/trophy_gunwon.jpg";
      case $TSB_GROUPHAT:
         return "assets/client/ui/hud/trophy_hatwon.jpg";
      case $TSB_GROUPGOLD:
         return "assets/client/ui/hud/trophy_goldwon.jpg";
      case $TSB_GROUPFOOD:
         return "assets/client/ui/hud/trophy_foodwon.jpg";
      case $TSB_GROUPSKULL:
         return "assets/client/ui/hud/trophy_skullwon.jpg";
      case $TSB_GROUPSHOE:
         return "assets/client/ui/hud/trophy_shoewon.jpg";
      case $TSB_GROUPBOOT:
         return "assets/client/ui/hud/trophy_bootwon.jpg";
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the badge name given the badge string id
//! \param %badgeId Badge string id
//! \retval string Badge name that will be displayed
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeName(%badgeId)
{
   %badgeNameId = %badgeId @ "_name";
   return (tsGetTaskString(%badgeNameId));
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the badge description given the badge string id
//! \param %badgeId Badge string id
//! \retval string Badge description that will be displayed
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeDescription(%badgeId)
{
   %badgeDescId = %badgeId @ "_desc";
   return (tsGetTaskString(%badgeDescId));
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the locked name for the badge
//! \retval string Locked name
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeLockedName()
{
   return (tsGetTaskString("badge_lock_name"));
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the locked description for the badge
//! \retval string Locked description
////////////////////////////////////////////////////////////////////////////////
function tsGetBadgeLockedDescription()
{
   return (tsGetTaskString("badge_lock_desc"));
}

////////////////////////////////////////////////////////////////////////////////
//! Renders all badges to the badge gui
////////////////////////////////////////////////////////////////////////////////
function tsRenderBadgeGui()
{
   slgRenderWorld(false);
   
   // Determine start and end render indexes
   %start = $TSB_CURRPAGE * $TSB_BADGEPERPAGE;
   %end   = %start + $TSB_BADGEPERPAGE;
   
   // Hide all gui elements on page first
   for(%i = 0; %i < $TSB_BADGEPERPAGE; %i++) {
      // Get element names
      %icon = $TSB_ICONGUI @ %i;
      %name = $TSB_NAMEGUI @ %i;
      %desc = $TSB_DESCGUI @ %i;
      %cmpl = $TSB_CMPLGUI @ %i;
      
      // Hide elements      
      if(isObject(%icon)) {
         %icon.Visible = false;
      }
      if(isObject(%name)) {
         %name.Visible = false;
      }      
      if(isObject(%desc)) {
         %desc.Visible = false;
      }
      if(isObject(%cmpl)) {
         %cmpl.Visible = false;
      }
   }
   
   // Render the individual badges
   tsRenderBadges(%start, %end);
}

////////////////////////////////////////////////////////////////////////////////
//! Renders an individual badge to the gui
//! \param %badgeId Badge string id
//! \param %groupId Group id for badge
//! \param %renderIdx Render index for badge
//! \param %locked True if badge is locked
//! \param %awarded True if badge is awarded
////////////////////////////////////////////////////////////////////////////////
function tsRenderBadge(%badgeId, %groupId, %renderIdx, %locked, %awarded)
{
   // Get render location for badge
   %renderLoc = %renderIdx - ($TSB_BADGEPERPAGE * $TSB_CURRPAGE);
   
   // Get render elements
   %icon = $TSB_ICONGUI @ %renderLoc;
   %name = $TSB_NAMEGUI @ %renderLoc;
   %desc = $TSB_DESCGUI @ %renderLoc;
   %cmpl = $TSB_CMPLGUI @ %renderLoc;
   
   // If the badge is locked then
   if(%locked) {
      // Get locked elements
      %icon.stateUp = tsGetBadgeLockedImage();
      %name.stateUp = tsGetBadgeLockedName();
      %desc.stateUp = tsGetBadgeLockedDescription();
   }
   
   // Otherwise 
   else {
      // If the badge has been awarded then
      if(%awarded) {
         // Get awarded elements
         %icon.stateUp = tsGetBadgeAwardImage(%groupId);
         %cmpl.Visible = true;
      }
      // Otherwise
      else {
         // Get normal elements
         %icon.stateUp = tsGetBadgeNormalImage(%groupId);
      }
      
      // Get the rest of the normal elements
      %name.stateUp = tsGetBadgeName(%badgeId);
      %desc.stateUp = tsGetBadgeDescription(%badgeId);
   }
   
   %icon.Visible = true;
   %name.Visible = true;
   %desc.Visible = true;
}

////////////////////////////////////////////////////////////////////////////////
//! Goes to and renders the next page
////////////////////////////////////////////////////////////////////////////////
function tsBadgeNextPage()
{
   // Go to next page if possible
   if($TSB_CURRPAGE + 1 < $TSB_MAXPAGES) {
      $TSB_CURRPAGE++;
      
      // Render current page
      tsRenderBadgeGui();
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Goes to and renders the previous page
////////////////////////////////////////////////////////////////////////////////
function tsBadgePrevPage()
{
   // Go to prev page if possible
   if($TSB_CURRPAGE > 0) {
      $TSB_CURRPAGE--;
      
      // Render current page
      tsRenderBadgeGui();
   }
}
// End cmn_badges.cs